<?php

/**
 * Language variables
 */
class NLBookingAPILang
{
    static protected $_Language = [
        // System
        'error_code' => 'Неизвестная языковая переменная %s',
        'error_loginNoEmpty' => 'Логин не может быть пустым',
        'error_APInoConnect' => 'Ошибка подключения',
        'error_emptyCredentials' => 'Установите логин и пароль',
        'error_noEmpty' => 'Значение %s не может быть пустым',
        'error_noEmptyAndInt' => 'Неверное значение %s, должно быть целое и больше 0',
        'error_noInt' => 'Неверное значение %s, должно быть целое',
        'error_date' => 'Неверный формат даты, %s. Правильный формат - "Y-m-d". Пример: 2020-01-01',
        'error_period' => 'Неверный формат даты начала или даты окончания, %s. Правильный формат - "Y-m-d". Пример: 2020-01-01',
        'error_periodValues' => 'Дата окончания периода должна быть больше даты начала',
        'error_time' => 'Неверный формат времени, %s. Правильный формат - "H:i". Пример: 18:10',
        'error_oneFromList' => 'Неверное значение %s, доступные значения: %s',
        // Set
        'error_setCurrency' => 'Установите валюту',
        'error_setCity' => 'Установите город',
        'error_setCityOrHotel' => 'Установите город или отель',
        'error_setHotel' => 'Установите отель',
        'error_setRoom' => 'Установите номер',
        'error_setHotelRatePlan' => 'Установите тарифный план',
        'error_setPlacementType' => 'Установите тип размещения',
        'error_setFoodType' => 'Установите тип питания',
        'error_setPeriod' => 'Установите период поиска',
        'error_setAdults' => 'Установите количество взрослых',
        'error_setChildrenAges' => 'Укажите возраст всех детей',
        'error_setValForHotel' => 'Установите значение %s для отеля',
        'error_setValForRoom' => 'Установите значение %s для номера',
        'error_setValForExcursion' => 'Установите значение %s для экскурсии',
        'error_setValForTransfer' => 'Установите значение %s для трансфера',
        'error_setValForOrder' => 'Установите значение %s для заказа',
        'error_setValForUnitOrder' => 'Установите значение %s для единичного заказа'
    ];

    /**
     * Line
     * @static
     * @param string $name
     * @param string $str
     * @return mixed|string
     */
    static public function line($name, $str = '')
    {
        return (isset(self::$_Language[$name])) ? (($str == '') ? self::$_Language[$name] : sprintf(self::$_Language[$name], $str)) : sprintf(self::$_Language['error_code'], $name);
    }
}

/**
 * Check input data
 */
class NLBookingAPICheck
{
    /**
     * Check on empty
     * @param string $checkData
     * @param string $alias
     * @param string $langKey
     * @return bool
     * @throws Exception
     */
    public function noEmpty($checkData = '', $alias = '', $langKey = '')
    {
        if (empty($checkData)) {
            $langKey = ($langKey == '') ? 'error_noEmpty' : $langKey;
            throw new Exception(NLBookingAPILang::line($langKey, $alias));
        }

        return true;
    }

    /**
     * Check on int
     * @param string $checkData
     * @param string $alias
     * @param string $langKey
     * @return bool
     * @throws Exception
     */
    public function noInt($checkData = '', $alias = '', $langKey = '')
    {
        if (!is_int($checkData)) {
            $langKey = ($langKey == '') ? 'error_noInt' : $langKey;
            throw new Exception(NLBookingAPILang::line($langKey, $alias . '="' . $checkData . '"'));
        }

        return true;
    }

    /**
     * Check on empty and int
     * @param string $checkData
     * @param string $alias
     * @param string $langKey
     * @return bool
     * @throws Exception
     */
    public function noEmptyAndInt($checkData = '', $alias = '', $langKey = '')
    {
        if (empty($checkData) || !is_int($checkData)) {
            $langKey = ($langKey == '') ? 'error_noEmptyAndInt' : $langKey;
            throw new Exception(NLBookingAPILang::line($langKey, $alias . '="' . $checkData . '"'));
        }

        return true;
    }

    /**
     * Check on date
     * @param string $date
     * @return bool
     * @throws Exception
     */
    public function date($date = '')
    {
        $time = strtotime($date);
        if ($date != date('Y-m-d', $time)) {
            throw new Exception(sprintf(NLBookingAPILang::line('error_date'), $date . '="' . $date . '"'));
        }

        return true;
    }

    /**
     * Check on period
     * @param string $from
     * @param string $to
     * @return bool
     * @throws Exception
     */
    public function period($from = '', $to = '')
    {
        $timeFrom = strtotime($from);
        $timeTo = strtotime($to);
        if ($from != date('Y-m-d', $timeFrom) || $to != date('Y-m-d', $timeTo)) {
            throw new Exception(sprintf(NLBookingAPILang::line('error_period'), $from . '="' . $from . '"', $to . '="' . $to . '"'));
        }
        if ($timeFrom >= $timeTo) {
            throw new Exception(NLBookingAPILang::line('error_periodValues'));
        }

        return true;
    }

    /**
     * Check on time
     * @param string $time
     * @param string $alias
     * @param string $langKey
     * @return bool
     * @throws Exception
     */
    public function time($time = '', $alias = '', $langKey = '')
    {
        if ($time != date('H:i', strtotime($time))) {
            $langKey = ($langKey == '') ? 'error_time' : $langKey;
            throw new Exception(NLBookingAPILang::line($langKey, $alias . '="' . $time . '"'));
        }

        return true;
    }

    /**
     * One from list check
     * @param string $element
     * @param array $list
     * @param string $alias
     * @param string $langKey
     * @return bool
     * @throws Exception
     */
    public function oneFromList($element = '', $list = [], $alias = '', $langKey = '')
    {
        if (!in_array($element, $list)) {
            $langKey = ($langKey == '') ? 'error_oneFromList' : $langKey;
            throw new Exception(sprintf(NLBookingAPILang::line($langKey), $alias . '="' . $element . '"', implode(', ', $list)));
        }

        return true;
    }
}

class NLBookingAPISearchParams
{
    protected $_params = [
        'Currency' => 0,
        'City' => 0,
        'DateRange' => [],
        'Adults' => []
    ];

    /**
     * Add currency
     * @param integer $idCurrency
     * @throws Exception
     */
    public function addCurrency($idCurrency = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idCurrency, '$idCurrency');

        $this->_params['Currency'] = $idCurrency;
    }

    /**
     * Add city
     * @param integer $idCity
     * @throws Exception
     */
    public function addCity($idCity = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idCity, '$idCity');

        $this->_params['City'] = $idCity;
    }

    /**
     * Add period
     * @param string $from
     * @param string $to
     * @throws Exception
     */
    public function addPeriod($from = '', $to = '')
    {
        NLBookingAPICheck::period($from, $to);

        $this->_params['DateRange'] = ['from' => $from, 'to' => $to];
    }

    /**
     * Add adults
     * @param integer $count
     * @throws Exception
     */
    public function addAdults($count = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($count, '$count');

        $this->_params['Adults'] = ['count' => $count];
    }

    /**
     * Get
     * @return array
     * @throws Exception
     */
    public function get()
    {
        // Check all needed params
        NLBookingAPICheck::noEmpty($this->_params['Currency'], '', 'error_setCurrency');
        NLBookingAPICheck::noEmpty($this->_params['City'], '', 'error_setCity');
        NLBookingAPICheck::noEmpty($this->_params['DateRange'], '', 'error_setPeriod');
        NLBookingAPICheck::noEmpty($this->_params['Adults'], '', 'error_setAdults');

        return $this->_params;
    }
}

class NLBookingAPISearchHotelsParams extends NLBookingAPISearchParams
{
    public function __construct()
    {
        $this->_params['Hotel'] = 0;
        $this->_params['Children'] = [];
        $this->_params['Availability'] = 'online';
        $this->_params['PriceDetail'] = 0;
        $this->_params['OneRoom'] = 0;
        $this->_params['Page'] = 0;
        $this->_params['PerPage'] = 0;
    }

    /**
     * Add hotel
     * @param integer $idHotel
     * @throws Exception
     */
    public function addHotel($idHotel = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idHotel, '$idHotel');

        $this->_params['Hotel'] = $idHotel;
    }

    /**
     * Add children
     * @param integer $count
     * @param array $ages
     * @throws Exception
     */
    public function addChildren($count = 0, $ages = [])
    {
        NLBookingAPICheck::noEmptyAndInt($count, '$count');

        if (empty($ages) || $count != count($ages)) {
            throw new Exception(NLBookingAPILang::line('error_setChildrenAges'));
        }

        $this->_params['Children'] = ['count' => $count];

        foreach ($ages as $age) {
            NLBookingAPICheck::noInt($age, '$age');

            $this->_params['Children']['Child'][] = ['age' => $age];
        }
    }

    /**
     * Set availability of rooms
     * Can be 'online', 'onrequest', 'all'
     * @param string $availability
     * @throws Exception
     */
    public function addAvailability($availability = 'online')
    {
        // prepare data
        $availability = strtolower($availability);

        $allowed = ['online', 'onrequest', 'all'];

        NLBookingAPICheck::oneFromList($availability, $allowed, '$availability');

        $this->_params['Availability'] = $availability;
    }

    /**
     * Add price detail
     * @param integer $priceDetail
     * @throws Exception
     */
    public function addPriceDetail($priceDetail = 0)
    {
        $allowed = [0, 1];

        NLBookingAPICheck::oneFromList($priceDetail, $allowed, '$priceDetail');

        $this->_params['PriceDetail'] = intval($priceDetail);
    }

    /**
     * Add one room
     * @param integer $oneRoom
     * @throws Exception
     */
    public function addOneRoom($oneRoom = 0)
    {
        $allowed = [0, 1];

        NLBookingAPICheck::oneFromList($oneRoom, $allowed, '$oneRoom');

        $this->_params['OneRoom'] = intval($oneRoom);
    }

    /**
     * Add page
     * @param integer $page
     * @throws Exception
     */
    public function addPage($page = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($page, '$page');

        $this->_params['Page'] = $page;
    }

    /**
     * Add per page
     * @param integer $perPage
     * @throws Exception
     */
    public function addPerPage($perPage = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($perPage, '$perPage');

        $this->_params['PerPage'] = $perPage;
    }

    /**
     * Get
     * @return array
     * @throws Exception
     */
    public function get()
    {
        // Check all needed params
        if (empty($this->_params['City']) && empty($this->_params['Hotel'])) {
            NLBookingAPICheck::noEmpty($this->_params['City'], '', 'error_setCityOrHotel');
        }

        NLBookingAPICheck::noEmpty($this->_params['Currency'], '', 'error_setCurrency');
        NLBookingAPICheck::noEmpty($this->_params['DateRange'], '', 'error_setPeriod');
        NLBookingAPICheck::noEmpty($this->_params['Adults'], '', 'error_setAdults');

        return $this->_params;
    }
}

class NLBookingAPIHotelOvertimeParams
{
    protected $_params = [
        'Currency' => 0,
        'Room' => 0,
        'HotelRatePlan' => 0,
        'PlacementType' => 0,
        'FoodType' => 0,
        'DateRange' => [],
        'EarlyCheckIn' => [],
        'LateCheckOut' => []
    ];

    /**
     * Add id currency
     * @param integer $idCurrency
     * @throws Exception
     */
    public function addCurrency($idCurrency = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idCurrency, '$idCurrency');

        $this->_params['Currency'] = $idCurrency;
    }

    /**
     * Add id room
     * @param integer $idRoom
     * @throws Exception
     */
    public function addRoom($idRoom = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idRoom, '$idRoom');

        $this->_params['Room'] = $idRoom;
    }

    /**
     * Add id hotel rate plan
     * @param integer $idHotelRatePlan
     * @throws Exception
     */
    public function addHotelRatePlan($idHotelRatePlan = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idHotelRatePlan, '$idHotelRatePlan');

        $this->_params['HotelRatePlan'] = $idHotelRatePlan;
    }

    /**
     * Add id placement type
     * @param integer $idPlacementType
     * @throws Exception
     */
    public function addPlacementType($idPlacementType = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idPlacementType, '$idPlacementType');

        $this->_params['PlacementType'] = $idPlacementType;
    }

    /**
     * Add id food type
     * @param integer $idFoodType
     * @throws Exception
     */
    public function addFoodType($idFoodType = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idFoodType, '$idFoodType');

        $this->_params['FoodType'] = $idFoodType;
    }

    /**
     * Add period
     * @param string $from
     * @param string $to
     * @throws Exception
     */
    public function addPeriod($from = '', $to = '')
    {
        NLBookingAPICheck::period($from, $to);

        $this->_params['DateRange'] = ['from' => $from, 'to' => $to];
    }

    /**
     * Add early check-in time
     * @param string $time
     * @throws Exception
     */
    public function addEarlyCheckInTime($time)
    {
        NLBookingAPICheck::time($time, '$time');

        $this->_params['EarlyCheckIn']['Time'] = $time;
    }

    /**
     * Add early check-in time from
     * @param string $timeFrom
     * @throws Exception
     */
    public function addEarlyCheckInTimeFrom($timeFrom)
    {
        NLBookingAPICheck::time($timeFrom, '$timeFrom');

        $this->_params['EarlyCheckIn']['TimeFrom'] = $timeFrom;
    }

    /**
     * Add early check-in time to
     * @param string $timeTo
     * @throws Exception
     */
    public function addEarlyCheckInTimeTo($timeTo)
    {
        NLBookingAPICheck::time($timeTo, '$timeTo');

        $this->_params['EarlyCheckIn']['TimeTo'] = $timeTo;
    }

    /**
     * Add late check-out time
     * @param string $time
     * @throws Exception
     */
    public function addLateCheckOutTime($time)
    {
        NLBookingAPICheck::time($time, '$time');

        $this->_params['LateCheckOut']['Time'] = $time;
    }

    /**
     * Add late check-out time from
     * @param string $timeFrom
     * @throws Exception
     */
    public function addLateCheckOutTimeFrom($timeFrom)
    {
        NLBookingAPICheck::time($timeFrom, '$timeFrom');

        $this->_params['LateCheckOut']['TimeFrom'] = $timeFrom;
    }

    /**
     * Add late check-out time to
     * @param string $timeTo
     * @throws Exception
     */
    public function addLateCheckOutTimeTo($timeTo)
    {
        NLBookingAPICheck::time($timeTo, '$timeTo');

        $this->_params['LateCheckOut']['TimeTo'] = $timeTo;
    }

    /**
     * Get all params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        // Check all needed params
        NLBookingAPICheck::noEmpty($this->_params['Currency'], '', 'error_setCurrency');
        NLBookingAPICheck::noEmpty($this->_params['Room'], '', 'error_setRoom');
        NLBookingAPICheck::noEmpty($this->_params['HotelRatePlan'], '', 'error_setHotelRatePlan');
        NLBookingAPICheck::noEmpty($this->_params['PlacementType'], '', 'error_setPlacementType');
        NLBookingAPICheck::noEmpty($this->_params['FoodType'], '', 'error_setFoodType');
        NLBookingAPICheck::noEmpty($this->_params['DateRange'], '', 'error_setPeriod');

        return $this->_params;
    }
}

class NLBookingAPIHotelSurchargesParams
{
    protected $_params = [
        'Currency' => 0,
        'Hotel' => 0
    ];

    /**
     * Add id currency
     * @param integer $idCurrency
     * @throws Exception
     */
    public function addCurrency($idCurrency = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idCurrency, '$idCurrency');

        $this->_params['Currency'] = $idCurrency;
    }

    /**
     * Add id hotel
     * @param integer $idHotel
     * @throws Exception
     */
    public function addHotel($idHotel = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idHotel, '$idHotel');

        $this->_params['Hotel'] = $idHotel;
    }

    /**
     * Get all params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        // Check all needed params
        NLBookingAPICheck::noEmpty($this->_params['Currency'], '', 'error_setCurrency');
        NLBookingAPICheck::noEmpty($this->_params['Hotel'], '', 'error_setHotel');

        return $this->_params;
    }
}

class NLBookingAPISearchOrdersParams
{
    protected $_params = [
        'DateFrom' => '',
        'DateTo' => '',
        'Hotel' => 0
    ];

    public function __construct()
    {
        $this->_params['DateFrom'] = date('Y-m-d');
    }

    /**
     * Add date from
     * @param string $dateFrom
     * @throws Exception
     */
    public function addDateFrom($dateFrom = '')
    {
        NLBookingAPICheck::date($dateFrom, '$dateFrom');

        $this->_params['DateFrom'] = $dateFrom;
    }

    /**
     * Add date to
     * @param string $dateTo
     * @throws Exception
     */
    public function addDateTo($dateTo = '')
    {
        NLBookingAPICheck::date($dateTo, '$dateTo');

        $this->_params['DateTo'] = $dateTo;
    }

    /**
     * Add hotel
     * @param integer $idHotel
     * @throws Exception
     */
    public function addHotel($idHotel = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idHotel, '$idHotel');

        $this->_params['Hotel'] = $idHotel;
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        // Check all needed params
        if (isset($this->_params['DateFrom']) && !empty($this->_params['DateFrom']) &&
                isset($this->_params['DateTo']) && !empty($this->_params['DateTo'])) {
            NLBookingAPICheck::period($this->_params['DateFrom'], $this->_params['DateTo']);
        }

        return $this->_params;
    }
}

class NLBookingAPIOrderParams
{
    protected $_params = [
        'Currency' => 0,
        'Hotel' => [],
        'Excursion' => [],
        'Transfer' => []
    ];

    /**
     * Add id currency
     * @param integer $idCurrency
     * @throws Exception
     */
    public function addCurrency($idCurrency = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idCurrency, '$idCurrency');

        $this->_params['Currency'] = $idCurrency;
    }

    /**
     * Add hotel
     * @param NLBookingAPIOrderHotelParams $NLBookingAPIOrderHotelParams
     * @throws Exception
     */
    public function addHotel(NLBookingAPIOrderHotelParams $NLBookingAPIOrderHotelParams)
    {
        $this->_params['Hotel'][] = $NLBookingAPIOrderHotelParams->get();
    }

    /**
     * Add excursion
     * @param NLBookingAPIOrderExcursionParams $NLBookingAPIOrderExcursionParams
     * @throws Exception
     */
    public function addExcursion(NLBookingAPIOrderExcursionParams $NLBookingAPIOrderExcursionParams)
    {
        $this->_params['Excursion'][] = $NLBookingAPIOrderExcursionParams->get();
    }

    /**
     * Add transfer
     * @param NLBookingAPIOrderTransferParams $NLBookingAPIOrderTransferParams
     * @throws Exception
     */
    public function addTransfer(NLBookingAPIOrderTransferParams $NLBookingAPIOrderTransferParams)
    {
        $this->_params['Transfer'][] = $NLBookingAPIOrderTransferParams->get();
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        NLBookingAPICheck::noEmpty($this->_params['Currency'], 'Currency', 'error_setValForOrder');

        return $this->_params;
    }
}

class NLBookingAPIEditOrderParams extends NLBookingAPIOrderParams
{
    public function __construct()
    {
        $this->_params['Order']['UniqueID'] = '';
    }

    /**
     * Set order unique id
     * @param string $uniqueID
     * @throws Exception
     */
    public function setOrder($uniqueID = '')
    {
        NLBookingAPICheck::noEmpty($uniqueID, '$uniqueID');

        $this->_params['Order']['UniqueID'] = $uniqueID;
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        NLBookingAPICheck::noEmpty($this->_params['Order']['UniqueID'], 'UniqueID', 'error_setValForOrder');

        return parent::get();
    }
}

class NLBookingAPIOrderHotelParams
{
    protected $_params = [
        'id' => 0,
        'DateRange' => [],
        'Room' => []
    ];

    /**
     * Add id hotel
     * @param integer $idHotel
     * @throws Exception
     */
    public function addHotel($idHotel = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idHotel, '$idHotel');

        $this->_params['id'] = $idHotel;
    }

    /**
     * Add period
     * @param string $from
     * @param string $to
     * @throws Exception
     */
    public function addPeriod($from = '', $to = '')
    {
        NLBookingAPICheck::period($from, $to);

        $this->_params['DateRange'] = ['from' => $from, 'to' => $to];
    }

    /**
     * Add room
     * @param NLBookingAPIOrderHotelRoomParams $NLBookingAPIOrderHotelRoomParams
     * @throws Exception
     */
    public function addRoom(NLBookingAPIOrderHotelRoomParams $NLBookingAPIOrderHotelRoomParams)
    {
        $this->_params['Room'][] = $NLBookingAPIOrderHotelRoomParams->get();
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        NLBookingAPICheck::noEmpty($this->_params['id'], 'id', 'error_setValForHotel');
        NLBookingAPICheck::noEmpty($this->_params['DateRange'], 'DateRange', 'error_setValForHotel');
        NLBookingAPICheck::noEmpty($this->_params['Room'], 'Room', 'error_setValForHotel');

        return $this->_params;
    }
}

class NLBookingAPIEditOrderHotelParams extends NLBookingAPIOrderHotelParams
{
    public function __construct()
    {
        $this->_params['UniqueID'] = '';
    }

    /**
     * Set order hotel unique id
     * @param string $uniqueID
     * @throws Exception
     */
    public function setOrderHotel($uniqueID = '')
    {
        NLBookingAPICheck::noEmpty($uniqueID, '$uniqueID');

        $this->_params['UniqueID'] = $uniqueID;
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        NLBookingAPICheck::noEmpty($this->_params['UniqueID'], 'UniqueID', 'error_setValForHotel');

        return parent::get();
    }
}

class NLBookingAPIOrderHotelRoomParams
{
    protected $_params = [
        'id' => 0,
        'HotelRatePlan' => 0,
        'PlacementType' => 0,
        'FoodType' => 0,
        'Amount' => []
    ];

    /**
     * Add id room
     * @param integer $idRoom
     * @throws Exception
     */
    public function addRoom($idRoom = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idRoom, '$idRoom');

        $this->_params['id'] = $idRoom;
    }

    /**
     * Add id hotel rate plan
     * @param integer $idHotelRatePlan
     * @throws Exception
     */
    public function addHotelRatePlan($idHotelRatePlan = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idHotelRatePlan, '$idHotelRatePlan');

        $this->_params['HotelRatePlan'] = $idHotelRatePlan;
    }

    /**
     * Add id placement type
     * @param integer $idPlacementType
     * @throws Exception
     */
    public function addPlacementType($idPlacementType = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idPlacementType, '$idPlacementType');

        $this->_params['PlacementType'] = $idPlacementType;
    }

    /**
     * Add food type
     * @param integer $idFoodType
     * @throws Exception
     */
    public function addFoodType($idFoodType = 0)
    {
        NLBookingAPICheck::noEmptyAndInt($idFoodType, '$idFoodType');

        $this->_params['FoodType'] = $idFoodType;
    }

    /**
     * Add amount
     * @param NLBookingAPIOrderHotelRoomAmountParams $NLBookingAPIOrderHotelRoomAmountParams
     * @throws Exception
     */
    public function addAmount(NLBookingAPIOrderHotelRoomAmountParams $NLBookingAPIOrderHotelRoomAmountParams)
    {
        $this->_params['Amount'][] = $NLBookingAPIOrderHotelRoomAmountParams->get();
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        NLBookingAPICheck::noEmpty($this->_params['id'], 'id', 'error_setValForRoom');
        NLBookingAPICheck::noEmpty($this->_params['HotelRatePlan'], 'HotelRatePlan', 'error_setValForRoom');
        NLBookingAPICheck::noEmpty($this->_params['PlacementType'], 'Placement', 'error_setValForRoom');
        NLBookingAPICheck::noEmpty($this->_params['FoodType'], 'FoodType', 'error_setValForRoom');
        NLBookingAPICheck::noEmpty($this->_params['Amount'], 'Amount', 'error_setValForRoom');

        return $this->_params;
    }
}

class NLBookingAPIOrderHotelRoomAmountParams
{
    protected $_params = [
        'Adult' => [],
        'Child' => [],
        'EarlyCheckIn' => [],
        'LateCheckOut' => [],
        'ClientNote' => ''
    ];

    /**
     * Add adult
     * @param string $name
     * @throws Exception
     */
    public function addAdult($name = '')
    {
        NLBookingAPICheck::noEmpty($name, '$name');

        $this->_params['Adult'][] = ['name' => $name];
    }

    /**
     * Add child
     * @param string $name
     * @param integer $age
     * @throws Exception
     */
    public function addChild($name = '', $age = 0)
    {
        NLBookingAPICheck::noEmpty($name, '$name');
        NLBookingAPICheck::noEmptyAndInt($age, '$age');

        $this->_params['Child'][] = ['name' => $name, 'age' => $age];
    }

    /**
     * Add early check-in
     * @param string $time
     * @param integer $type
     * @throws Exception
     */
    public function addEarlyCheckIn($time = '', $type = 1)
    {
        NLBookingAPICheck::time($time, '$time');

        $this->_params['EarlyCheckIn'] = ['time' => $time, 'type' => intval($type)];
    }

    /**
     * Add late check-out
     * @param string $time
     * @throws Exception
     */
    public function addLateCheckOut($time = '')
    {
        NLBookingAPICheck::time($time, '$time');

        $this->_params['LateCheckOut'] = ['time' => $time];
    }

    /**
     * Add client note
     * @param string $clientNote
     * @throws Exception
     */
    public function addClientNote($clientNote = '')
    {
        $this->_params['ClientNote'] = $clientNote;
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        NLBookingAPICheck::noEmpty($this->_params['Adult'], '', 'error_setAdults');

        return $this->_params;
    }
}

abstract class NLBookingAPIOrderAddService
{
    protected $_params = [
        'id' => 0,
        'DateRange' => [],
        'Persons' => 0,
        'Time' => '',
        'Note' => '',
        'MeetingPlace' => ''
    ];

    /**
     * Add period
     * @param string $from
     * @param string $to
     * @throws Exception
     */
    public function addPeriod($from = '', $to = '')
    {
        NLBookingAPICheck::period($from, $to);

        $this->_params['DateRange'] = ['from' => $from, 'to' => $to];
    }

    /**
     * Add persons
     * @param string $persons
     * @throws Exception
     */
    public function addPersons($persons = '')
    {
        NLBookingAPICheck::noEmptyAndInt($persons, '$persons');

        $this->_params['Persons'] = $persons;
    }

    /**
     * Add time
     * @param string $time
     * @throws Exception
     */
    public function addTime($time = '')
    {
        NLBookingAPICheck::time($time, '$time');

        $this->_params['Time'] = $time;
    }

    /**
     * Add meeting place
     * @param string $meetingPlace
     * @throws Exception
     */
    public function addMeetingPlace($meetingPlace = '')
    {
        NLBookingAPICheck::noEmpty($meetingPlace, '$meetingPlace');

        $this->_params['MeetingPlace'] = $meetingPlace;
    }

    /**
     * Add note
     * @param string $note
     * @throws Exception
     */
    public function addNote($note = '')
    {
        NLBookingAPICheck::noEmpty($note, '$note');

        $this->_params['Note'] = $note;
    }

    /**
     * Get params
     * @param string $langKey
     * @return array
     * @throws Exception
     */
    public function get($langKey)
    {
        NLBookingAPICheck::noEmpty($this->_params['id'], 'id', $langKey);
        NLBookingAPICheck::noEmpty($this->_params['DateRange'], 'DateRange', $langKey);
        NLBookingAPICheck::noEmpty($this->_params['Persons'], 'Persons', $langKey);
        NLBookingAPICheck::noEmpty($this->_params['Time'], 'Time', $langKey);
        NLBookingAPICheck::noEmpty($this->_params['MeetingPlace'], 'MeetingPlace', $langKey);

        return $this->_params;
    }
}

class NLBookingAPIOrderExcursionParams extends NLBookingAPIOrderAddService
{
    /**
     * Add id excursion
     * @param string $idExcursion
     * @throws Exception
     */
    public function addExcursion($idExcursion = '')
    {
        NLBookingAPICheck::noEmptyAndInt($idExcursion, '$idExcursion');

        $this->_params['id'] = $idExcursion;
    }

    /**
     * Get param
     * @param string $langKey
     * @return array
     * @throws Exception
     */
    public function get($langKey)
    {
        $langKey = 'error_setValForExcursion';
        return parent::get($langKey);
    }
}

class NLBookingAPIOrderTransferParams extends NLBookingAPIOrderAddService
{
    /**
     * Add id transfer
     * @param string $idTransfer
     * @throws Exception
     */
    public function addTransfer($idTransfer = '')
    {
        NLBookingAPICheck::noEmptyAndInt($idTransfer, '$idTransfer');

        $this->_params['id'] = $idTransfer;
    }

    /**
     * Add delivery price
     * @param string $deliveryPlace
     * @throws Exception
     */
    public function addDeliveryPlace($deliveryPlace = '')
    {
        NLBookingAPICheck::noEmpty($deliveryPlace, '$deliveryPlace');

        $this->_params['DeliveryPlace'] = $deliveryPlace;
    }

    /**
     * Get param
     * @param string $langKey
     * @return array
     * @throws Exception
     */
    public function get($langKey)
    {
        $langKey = 'error_setValForTransfer';
        NLBookingAPICheck::noEmpty($this->_params['DeliveryPlace'], 'DeliveryPlace', $langKey);

        return parent::get($langKey);
    }
}

class NLBookingAPIOrderInfoParams
{
    protected $_params = [
        'Order' => [
            'UniqueID' => ''
        ]
    ];

    /**
     * Set order unique id
     * @param string $uniqueID
     * @throws Exception
     */
    public function setOrder($uniqueID = '')
    {
        NLBookingAPICheck::noEmpty($uniqueID, '$uniqueID');

        $this->_params['Order']['UniqueID'] = $uniqueID;
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        NLBookingAPICheck::noEmpty($this->_params['Order']['UniqueID'], 'UniqueID', 'error_setValForOrder');

        return $this->_params;
    }
}

class NLBookingAPIActionOrderParams extends NLBookingAPIOrderInfoParams
{
    public function __construct()
    {
        $this->_params['Order']['OrderHotel'] = [];
        $this->_params['Order']['OrderExcursion'] = [];
        $this->_params['Order']['OrderTransfer'] = [];
    }

    /**
     * Set order hotel
     * @param NLBookingAPIActionOrderUnitParams $NLBookingAPIActionOrderUnitParams
     * @throws Exception
     */
    public function setOrderHotel(NLBookingAPIActionOrderUnitParams $NLBookingAPIActionOrderUnitParams)
    {
        $this->_params['Order']['OrderHotel'][] = $NLBookingAPIActionOrderUnitParams->get();
    }

    /**
     * Set order excursion
     * @param NLBookingAPIActionOrderUnitParams $NLBookingAPIActionOrderUnitParams
     * @throws Exception
     */
    public function setOrderExcursion(NLBookingAPIActionOrderUnitParams $NLBookingAPIActionOrderUnitParams)
    {
        $this->_params['Order']['OrderExcursion'][] = $NLBookingAPIActionOrderUnitParams->get();
    }

    /**
     * Set order transfer
     * @param NLBookingAPIActionOrderUnitParams $NLBookingAPIActionOrderUnitParams
     * @throws Exception
     */
    public function setOrderTransfer(NLBookingAPIActionOrderUnitParams $NLBookingAPIActionOrderUnitParams)
    {
        $this->_params['Order']['OrderTransfer'][] = $NLBookingAPIActionOrderUnitParams->get();
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        return parent::get();
    }
}

class NLBookingAPIActionOrderUnitParams
{
    protected $_params = [
        'UniqueID' => ''
    ];

    /**
     * Set unique id of unit order
     * @param string $uniqueID
     * @throws Exception
     */
    public function setUniqueID($uniqueID = '')
    {
        NLBookingAPICheck::noEmpty($uniqueID, '$uniqueID');

        $this->_params['UniqueID'] = $uniqueID;
    }

    /**
     * Get params
     * @return array
     * @throws Exception
     */
    public function get()
    {
        NLBookingAPICheck::noEmpty($this->_params['UniqueID'], 'UniqueID', 'error_setValForUnitOrder');

        return $this->_params;
    }
}

/**
 * Main Booking API class
 */
class NLBookingAPI
{
    protected $_version = '3.1';
    protected $_wsdl = 'http://apitest.newlogic.ua/api/v3/wsdl/';
    protected $_Credentials = [];
    protected $_Language = 'en';
    protected $_debug = false;
    protected $_connect = null;

    /**
     * Get current API version
     * @return string
     */
    public function getVersion()
    {
        return $this->_version;
    }

    /**
     * Enable/Disable debug
     * @param boolean $value
     */
    public function setDebug($value = false)
    {
        $this->_debug = $value;
    }

    /**
     * Set wsdl file location
     * @param string $location
     * @return void
     */
    public function setWSDL($location)
    {
        $this->_wsdl = $location;
    }

    /**
     * Set user login and password
     * @param string $login
     * @param string $password
     * @throws Exception
     */
    public function setCredentials($login = '', $password = '')
    {
        NLBookingAPICheck::noEmpty($login, '', 'error_loginNoEmpty');

        $this->_Credentials = [
            'Login' => $login,
            'Password' => $password
        ];
    }

    /**
     * Set request language
     * @param string $language
     * @throws Exception
     */
    public function setLanguage($language = 'en')
    {
        $this->_Language = $language;
    }

    /**
     * Connect to service
     * @return object
     * @throws Exception
     */
    public function getConnect()
    {
        if (is_null($this->_connect)) {
            $this->_connect = (!$this->_debug) ? new SoapClient($this->_wsdl) : new SoapClient($this->_wsdl, ["trace" => 1]);
        }

        if (is_null($this->_connect)) {
            throw new Exception(NLBookingAPILang::line('error_APInoConnect'));
        }

        return $this->_connect;
    }

    /**
     * Get countries
     * @return object
     * @throws Exception
     */
    public function getCountryList()
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ];
        $connect = $this->getConnect();

        return $connect->GetCountryList($params);
    }

    /**
     * Get regions of the country
     * @param integer $idCountry
     * @return object
     * @throws Exception
     */
    public function getRegionList($idCountry = 0)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');
        NLBookingAPICheck::noEmptyAndInt($idCountry, '$idCountry');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language,
            'Country' => $idCountry
        ];
        $connect = $this->getConnect();

        return $connect->GetRegionList($params);
    }

    /**
     * Get cities of the region
     * @param integer $idCountry
     * @param integer $idRegion
     * @return object
     * @throws Exception
     */
    public function getCityList($idCountry = 0, $idRegion = 0)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');
        NLBookingAPICheck::noEmptyAndInt($idCountry, '$idCountry');
        NLBookingAPICheck::noInt($idRegion, '$idRegion');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language,
            'Country' => $idCountry,
            'Region' => $idRegion
        ];
        $connect = $this->getConnect();

        return $connect->GetCityList($params);
    }

    /**
     * Get hotels of the city
     * @param integer $idCity
     * @return object
     * @throws Exception
     */
    public function getHotelList($idCity = 0)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');
        NLBookingAPICheck::noEmptyAndInt($idCity, '$idCity');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language,
            'City' => $idCity
        ];
        $connect = $this->getConnect();

        return $connect->GetHotelList($params);
    }

    /**
     * Get rooms of the hotel
     * @param integer $idHotel
     * @return object
     * @throws Exception
     */
    public function getRoomList($idHotel = 0)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');
        NLBookingAPICheck::noEmptyAndInt($idHotel, '$idHotel');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language,
            'Hotel' => $idHotel
        ];
        $connect = $this->getConnect();

        return $connect->GetRoomList($params);
    }

    /**
     * Get hotel classes
     * @return object
     * @throws Exception
     */
    public function getHotelClassList()
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ];
        $connect = $this->getConnect();

        return $connect->GetHotelClassList($params);
    }

    /**
     * Get room types
     * @return object
     * @throws Exception
     */
    public function getRoomTypeList()
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ];
        $connect = $this->getConnect();

        return $connect->GetRoomTypeList($params);
    }

    /**
     * Get room categories
     * @return object
     * @throws Exception
     */
    public function getRoomCategoryList()
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ];
        $connect = $this->getConnect();

        return $connect->GetRoomCategoryList($params);
    }

    /**
     * Get placement types
     * @return object
     * @throws Exception
     */
    public function getPlacementTypeList()
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ];
        $connect = $this->getConnect();

        return $connect->GetPlacementTypeList($params);
    }

    /**
     * Get food types
     * @return object
     * @throws Exception
     */
    public function getFoodTypeList()
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ];
        $connect = $this->getConnect();

        return $connect->GetFoodTypeList($params);
    }

    /**
     * Get excursions in the city
     * @param integer $idCity
     * @return object
     * @throws Exception
     */
    public function getExcursionList($idCity = 0)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');
        NLBookingAPICheck::noEmptyAndInt($idCity, '$idCity');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language,
            'City' => $idCity
        ];
        $connect = $this->getConnect();

        return $connect->GetExcursionList($params);
    }

    /**
     * Get transfers in the city
     * @param integer $idCity
     * @return object
     * @throws Exception
     */
    public function getTransferList($idCity = 0)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');
        NLBookingAPICheck::noEmptyAndInt($idCity, '$idCity');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language,
            'City' => $idCity
        ];
        $connect = $this->getConnect();

        return $connect->GetTransferList($params);
    }

    /**
     * Get currencies
     * @return object
     * @throws Exception
     */
    public function getCurrencyList()
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ];
        $connect = $this->getConnect();

        return $connect->GetCurrencyList($params);
    }

    /**
     * Get order statuses
     * @return object
     * @throws Exception
     */
    public function getOrderStatusList()
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = [
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ];
        $connect = $this->getConnect();

        return $connect->GetOrderStatusList($params);
    }

    /**
     * Search hotels
     * @param NLBookingAPISearchHotelsParams $searchParams
     * @return object
     * @throws Exception
     */
    public function searchHotels(NLBookingAPISearchHotelsParams $searchParams)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ], $searchParams->get());
        $connect = $this->getConnect();

        return $connect->SearchHotels($params);
    }

    /**
     * Get overtime
     * @param NLBookingAPIHotelOvertimeParams $hotelOvertime
     * @return mixed
     * @throws Exception
     */
    public function getHotelOvertime(NLBookingAPIHotelOvertimeParams $hotelOvertime)
    {
        // prepare data
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ], $hotelOvertime->get());
        $connect = $this->getConnect();

        return $connect->GetHotelOvertime($params);
    }

    /**
     * Get hotel surcharges
     * @param NLBookingAPIHotelSurchargesParams $hotelSurcharges
     * @return mixed
     * @throws Exception
     */
    public function getHotelSurcharges(NLBookingAPIHotelSurchargesParams $hotelSurcharges)
    {
        // prepare data
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Credentials' => $this->_Credentials,
            'Language' => $this->_Language
        ], $hotelSurcharges->get());
        $connect = $this->getConnect();

        return $connect->GetHotelSurcharges($params);
    }

    /**
     * Search excursions
     * @param NLBookingAPISearchParams $searchParams
     * @return mixed
     * @throws Exception
     */
    public function searchExcursions(NLBookingAPISearchParams $searchParams)
    {
        // prepare data
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Language' => $this->_Language,
            'Credentials' => $this->_Credentials
        ], $searchParams->get());
        $connect = $this->getConnect();

        return $connect->SearchExcursions($params);
    }

    /**
     * Search transfers
     * @param NLBookingAPISearchParams $searchParams
     * @return mixed
     * @throws Exception
     */
    public function searchTransfers(NLBookingAPISearchParams $searchParams)
    {
        // prepare data
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $searchData = $searchParams->get();
        $searchData['Date'] = $searchData['DateRange']['from'];
        $params = array_merge([
            'Language' => $this->_Language,
            'Credentials' => $this->_Credentials
        ], $searchData);
        $connect = $this->getConnect();

        return $connect->SearchTransfers($params);
    }

    /**
     * Add order
     * @param NLBookingAPIOrderParams $orderParams
     * @return mixed
     * @throws Exception
     */
    public function addOrder(NLBookingAPIOrderParams $orderParams)
    {
        // prepare data
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Language' => $this->_Language,
            'Credentials' => $this->_Credentials
        ], $orderParams->get());
        $connect = $this->getConnect();

        return $connect->AddOrder($params);
    }

    /**
     * Search orders
     * @param NLBookingAPISearchOrdersParams $searchParams
     * @return mixed
     * @throws Exception
     */
    public function searchOrders(NLBookingAPISearchOrdersParams $searchParams)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Language' => $this->_Language,
            'Credentials' => $this->_Credentials
        ], $searchParams->get());
        $connect = $this->getConnect();

        return $connect->SearchOrders($params);
    }

    /**
     * Get order information
     * @param NLBookingAPIOrderInfoParams $orderParams
     * @return mixed
     * @throws Exception
     */
    public function getOrder(NLBookingAPIOrderInfoParams $orderParams)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Language' => $this->_Language,
            'Credentials' => $this->_Credentials
        ], $orderParams->get());
        $connect = $this->getConnect();

        return $connect->GetOrder($params);
    }

    /**
     * Editing order
     * @param NLBookingAPIEditOrderParams $orderParams
     * @return mixed
     * @throws Exception
     */
    public function editOrder(NLBookingAPIEditOrderParams $orderParams)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Language' => $this->_Language,
            'Credentials' => $this->_Credentials
        ], $orderParams->get());
        $connect = $this->getConnect();

        return $connect->EditOrder($params);
    }

    /**
     * Simulate editing order
     * @param NLBookingAPIEditOrderParams $orderParams
     * @return mixed
     * @throws Exception
     */
    public function simulateEditOrder(NLBookingAPIEditOrderParams $orderParams)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Language' => $this->_Language,
            'Credentials' => $this->_Credentials
        ], $orderParams->get());
        $connect = $this->getConnect();

        return $connect->SimulateEditOrder($params);
    }

    /**
     * Confirm order
     * @param NLBookingAPIActionOrderParams $orderParams
     * @return mixed
     * @throws Exception
     */
    public function confirmOrder(NLBookingAPIActionOrderParams $orderParams)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Language' => $this->_Language,
            'Credentials' => $this->_Credentials
        ], $orderParams->get());
        $connect = $this->getConnect();

        return $connect->ConfirmOrder($params);
    }

    /**
     * Cancel order
     * @param NLBookingAPIActionOrderParams $orderParams
     * @return mixed
     * @throws Exception
     */
    public function cancelOrder(NLBookingAPIActionOrderParams $orderParams)
    {
        NLBookingAPICheck::noEmpty($this->_Credentials, '', 'error_emptyCredentials');

        $params = array_merge([
            'Language' => $this->_Language,
            'Credentials' => $this->_Credentials
        ], $orderParams->get());
        $connect = $this->getConnect();

        return $connect->CancelOrder($params);
    }
}
